local pprint = require "pprint"
local QueueMap = require "queue-map"
local the_eye = require "the-eye"

-- knowledge base
local kb = QueueMap.new()

for _, pixel in ipairs(the_eye) do
    kb:enqueue("image $image $x $y $r $g $b", {"eye", pixel[4], pixel[5], pixel[1], pixel[2], pixel[3]})
end

local function io_add(kb)
    local tuple = kb:peek "@add"
    if not tuple then return false end

    kb:poll "@add"
    kb:enqueue(tuple[1], tuple[2] + tuple[3])
    return true
end

local function io_put_pixel(kb)
    local tuple = kb:peek "@put pixel $r $g $b $x $y"
    if not tuple then return false end
    kb:poll "@put pixel $r $g $b $x $y"

    love.graphics.setColor(tuple[1] / 255, tuple[2] / 255, tuple[3] / 255, 1.0)     
    love.graphics.points(tuple[4], tuple[5]) 
    return true
end

local function put_pixel(kb)
    local put_pixel = kb:peek "put pixel $r $g $b"
    if not put_pixel then return false end
   
    local px = kb:peek "pixel x is $x"
    if not px then return false end


    local py = kb:peek "pixel y is $y"
    if not py then return false end
    
    kb:poll "put pixel $r $g $b"
    kb:poll "pixel x is $x"
    kb:poll "pixel y is $y"

    kb:enqueue("@put pixel $r $g $b $x $y", { put_pixel[1], put_pixel[2], put_pixel[3], px, py })
    return true
end

local function next_cursor(kb)
    local cx = kb:peek "next cursor x is $x"
    if not cx then return false end

    local cy = kb:peek "next cursor y is $y"
    if not cy  then return false end 

    kb:poll "next cursor x is $x"
    kb:poll "next cursor y is $y"

    kb:enqueue("cursor $x $y", {cx, cy})
end

local function draw_image_1(kb)
    local draw_image = kb:peek "draw image $image $x $y"
    if not draw_image then return end

    local cursor = kb:peek "cursor $x $y"
    if not cursor then return end 

    local image = kb:peek "image $image $x $y $r $g $b"
    if not image then return end

    kb:poll "draw image $image $x $y"
    kb:poll "cursor $x $y"
    kb:poll "image $image $x $y $r $g $b"

    kb:enqueue("put pixel $r $g $b", {image[4], image[5], image[6]})
    kb:enqueue("advance cursor $x $y", {cursor[1], cursor[2]})
    kb:enqueue("@add", {"pixel x is $x", draw_image[2], image[2]})
    kb:enqueue("@add", {"pixel y is $y", draw_image[3], image[3]})
    kb:enqueue("next draw $image $x $y", draw_image)
    kb:enqueue("image $image $x $y $r $g $b", image)
    return true
end

local function advance_cursor_1(kb)
    local tuple = kb:peek "advance cursor $x $y"
    if not tuple or tuple[1] ~= 249 or tuple[2] ~= 165 then return false end

    kb:poll "advance cursor $x $y"
    return true
end

local function advance_cursor_2(kb)
    local advance = kb:peek "advance cursor $x $y" 
    if not advance or advance[1] ~= 249 then return false end

    local next_draw_image = kb:peek "next draw $image $x $y"
    if not next_draw_image then return false end

    kb:poll "advance cursor $x $y"
    kb:poll "next draw $image $x $y"

    kb:enqueue("draw image $image $x $y", next_draw_image)
    kb:enqueue("next cursor x is $x", 0)
    kb:enqueue("@add", {"next cursor y is $y", 1, advance[2]})
    return true
end

local function advance_cursor_3(kb)
    local tuple = kb:peek "advance cursor $x $y" 
    if not tuple then return false end
    
    local next_draw_image = kb:peek "next draw $image $x $y"
    if not next_draw_image then return false end

    kb:poll "advance cursor $x $y"
    kb:poll "next draw $image $x $y"

    kb:enqueue("draw image $image $x $y", next_draw_image)
    kb:enqueue("@add", {"next cursor x is $x", 1, tuple[1]})
    kb:enqueue("next cursor y is $y", tuple[2])
    return true
end 

local function draw_image_2(kb)
    local tuple = kb:peek "draw image $image $x $y"
    if not tuple then return false end

    kb:poll "draw image $image $x $y"
    return true
end

local function tick(kb)
    local tuple = kb:peek "tick $dt"
    if not tuple then return false end
    
    kb:poll "tick $dt"
    
    kb:enqueue("draw image $image $x $y", {"eye", 0, 0})
    kb:enqueue("cursor $x $y", {0, 0})
    return true
end

local rules = 
    { io_add
    , io_put_pixel
    , put_pixel
    , next_cursor
    , draw_image_1
    , advance_cursor_1, advance_cursor_2, advance_cursor_3
    , draw_image_2
    , tick
    }

local screen = love.graphics.newCanvas()


function love.draw()
    love.graphics.setColor(1, 1, 1)
    love.graphics.draw(screen, 0, 0)
    love.graphics.setColor(1, 0, 0)
    love.graphics.print(tostring(love.timer.getFPS()) .. " fps", 0, 0)
end

function love.update(dt)
    local dt_fact = tostring(dt)
    love.graphics.setCanvas(screen)
    love.graphics.clear(0, 0, 0, 0)
    kb:enqueue("tick $dt", dt)
    local rewrite = true
    while rewrite do
        rewrite = false
        for i, rule in ipairs(rules) do
            if rule(kb) then
                rewrite = true
                break
            end
        end 
    end
    love.graphics.setCanvas()
end