extends Node
class_name _SimpleState

signal state_changed(name, state)

var state := { }


func register(state_name: String, inital_state: Dictionary = {}) -> void:
	state[state_name] = inital_state


func replace(state_name: String, new_state: Dictionary) -> void:
	state[state_name] = new_state.duplicate(true)
	emit_signal("state_changed", state_name, state)


func update(state_name: String, new_state: Dictionary) -> void:
	state[state_name] = _SimpleState.merge(get_state(state_name), new_state)
	emit_signal("state_changed", state_name, state)


func update_deep(state_name: String, new_state: Dictionary) -> void:
	state[state_name] = _SimpleState.merge_deep(get_state(state_name), new_state)
	emit_signal("state_changed", state_name, state)


func update_mut(state_name: String, new_state: Dictionary) -> void:
	_SimpleState.copy_to(state[state_name], new_state)
	emit_signal("state_changed")


func _on_update_state(data: Dictionary) -> void:
	update(data.state_name, data.state)


func get_state(state_name: String) -> Dictionary:
	return state[state_name].duplicate()


static func copy_to(a: Dictionary, b: Dictionary) -> void:
	for key in b:
		a[key] = b[key]



static func copy_to_deep(a: Dictionary, b: Dictionary) -> void:
	for k in b:
		if process_recursively(a, b, k):
			copy_to_deep(a[k], b[k])
		else:
			a[k] = b[k]


static func process_recursively(a, b, k) -> bool:
	return a.has(k) and a[k] is Dictionary and b[k] is Dictionary

static func merge(a: Dictionary, b: Dictionary) -> Dictionary:
	var c := a.duplicate(true)
	copy_to(c, b)
	return c


static func merge_deep(a: Dictionary, b: Dictionary) -> Dictionary:
	var c := a.duplicate(true)
	copy_to_deep(c, b)
	return c
