extends Control
class_name EventPlayer

# Signal
signal fade_in_started()
signal fade_out_started()
signal scene_ended()

# Proxied Signals
signal minigame_started(minigame : String, initial_state : Dictionary)
signal sequence_changed(sequence : String)

# Signals to Yield on
signal fade_finished()			# warning-ignore:unused_signal
signal minigame_finished()		# warning-ignore:unused_signal
signal change_scene_finished()	# warning-ignore:unused_signal

enum NextFade { FADE_OUT, FADE_IN }
var fade_active = false
var next_fade : NextFade = NextFade.FADE_OUT

@onready var playback             : Playback = $Playback
@onready var dialogue_presenter   = $DialoguePresenter
@onready var background_presenter = $BackgroundPresenter
@onready var character_handler    = $CharacterHandler
@onready var blackout_panel       = $BlackoutPanel
@onready var dialogue_box         = $DialogueBox
@onready var prize_screen         = $PrizeScreen
@onready var choice_panel         = get_tree().current_scene.get_node("%MainLayout/%ChoicePanel")

func _ready() -> void:
	playback.line_changed.connect(_on_line_changed)
	playback.dialogue_shown.connect(_on_dialogue_shown)
	playback.options_shown.connect(_on_options_shown)
	playback.dialogue_cleared.connect(_on_dialogue_cleared)
	playback.scene_finished.connect(_on_scene_finished)
	choice_panel.option_picked.connect(_on_option_picked)

	playback.extra_game_states = [ self
		, dialogue_presenter
		, background_presenter
		, character_handler
		, dialogue_box
		, prize_screen
		]


# Callbacks
func _on_line_changed(line_id: String) -> void:
	SimpleState.update("sequence",
		{ current_line_id = line_id })


func _on_dialogue_shown(dialogue_line : DialogueLine) -> void:
	if get_parent()._changing_scenes:
		await self.change_scene_finished

	await dialogue_presenter.show_dialogue(dialogue_line)
	playback.dialogue_finished.emit()

func _on_options_shown(options: Array[String]) -> void:
	if get_parent()._changing_scenes:
		await self.change_scene_finished

	choice_panel.show_options(options)


func _on_dialogue_cleared() -> void:
	dialogue_presenter.clear_dialogue()


func _on_scene_finished() -> void:
	emit_signal("scene_ended")


func _on_options_cleared() -> void:
	choice_panel.clear_options()


func _on_option_picked(__, line_id) -> void:
	playback.emit_signal("options_picked", line_id)


# Public Methods
func run(sequence: String, entry: String) -> void:
	playback.run(Events.items[sequence], entry)


func restore(sequence: String, entry: String) -> void:
	var dialogue_pos     = SimpleState.state.sequence.dialogue_pos
	var background       = SimpleState.state.sequence.background
	var blackout_visible = SimpleState.state.sequence.blackout_visible
	var characters       = SimpleState.state.sequence.characters

	next_fade            = SimpleState.state.sequence.next_fade

	dialogue_box.call(dialogue_pos)
	background_presenter.background(background)

	for character in characters:
		character_handler.enter(character, characters[character])

	choice_panel.clear_options()
	blackout(blackout_visible)
	playback.run(Events.items[sequence], entry, true)


func goto(sequence: String) -> void:
	sequence_changed.emit(sequence)
	await change_scene_finished


func visit_random_minigame() -> void:
	var seen = MemoryState.get_seen_minigames()
	var unseen = ArrayUtil.difference(CarnivalGames.items.keys(), seen)
	var key = ArrayUtil.either(unseen)
	var sequence_name = to_squence_name(key)

	sequence_changed.emit("minigames/%s/init" % sequence_name)


func to_squence_name(key) -> String:
	return key \
		.capitalize() \
		.replace(" ", "-") \
		.to_lower()


func exit() -> void:
	await fade_out()


func blackout(visibility: bool) -> void:
	SimpleState.update("sequence", { blackout_visible = visibility })
	blackout_panel.visible = visibility


func swap_background(layers: Array) -> void:
	await fade_out()
	await get_tree().process_frame
	background_presenter.background(layers)
	await fade_in()


func unimplemented() -> void:
	assert(false, "Unimplemented Scene")


func play_minigame(minigame: String, initial_state: Dictionary = {}) -> void:
	minigame_started.emit(minigame, initial_state)
	var result = await self.minigame_finished
	SimpleState.update_deep("temp", { values = result })



func fade() -> void:
	if fade_active: return

	fade_active = true

	match next_fade:
		NextFade.FADE_OUT:
			next_fade = NextFade.FADE_IN
			await fade_out()

		NextFade.FADE_IN:
			next_fade = NextFade.FADE_OUT
			await fade_in()

	SimpleState.update("sequence", { next_fade = next_fade })
	fade_active = false



func fade_out() -> void:
	fade_out_started.emit()
	await self.fade_finished


func fade_in() -> void:
	fade_in_started.emit()
	await self.fade_finished


func set_input_disabled(value: bool) -> void:
	$Background.input_disabled = value
	$Background/SubViewport.gui_disable_input = value


func goto_birth_scene() -> void:
	var scene = TempState["parasite"].birthing_scene
	ExploreState.emit_signal("event_triggered", scene)


func interaction(background: String) -> void:
	set_input_disabled(false)
	await background_presenter.start_interaction(background)
	set_input_disabled(true)
