from dataclasses import dataclass
from enum import Enum
from typing import Any, Self
import re

text_rendered_todo_re = re.compile(r"\[[ X/]{0,1}\]")


class TodoState(Enum):
    UNSTARTED = ("Unstarted", "[ ]")
    WORK_IN_PROGRESS = ("Work in Progress", "[/]")
    FINISHED = ("Finished", "[X]")

    def __init__(self, state: str, rendered: str):
        self.state = state
        self.rendered = rendered

    @classmethod
    def _missing_(cls, value: object) -> Any:
        if type(value) is str:
            value = "[ ]" if value == "[]" else value
            is_rendered = text_rendered_todo_re.match(value)

            for member in cls:
                if is_rendered and member.rendered == value:
                    return member
                elif member.state == value:
                    return member

        return super()._missing_(value)


class TodoPriority(Enum):
    LOW = "low"
    MID = "mid"
    HIGH = "high"

    def __init__(self, priority: str):
        self.priority = priority


todo_line_re = re.compile(
    r"^(?P<priority>(low|mid|high))\t(?P<state>\[[ X/]{0,1}\]) (?P<description>.*)$"
)


@dataclass
class TodoItem:
    state: TodoState
    priority: TodoPriority
    description: str

    @classmethod
    def from_string(cls, line: str) -> Self:
        if match := todo_line_re.match(line):
            state = TodoState(match.group("state"))
            priority = TodoPriority(match.group("priority"))
            return TodoItem(state, priority, match.group("description"))


if __name__ == "__main__":
    import sys

    todos = []

    for line in sys.stdin:
        if item := TodoItem.from_string(line.strip()):
            todos.append(item)

    print(f"Found {len(todos)} todo items.")

    for item in todos:
        print(item)